<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Account;
use App\Models\AccountType;
use App\Models\AccountTransaction;
use App\Models\Media;
use App\Models\TransactionPayment;
use App\Utils\Util;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
class AccountController extends BaseController
{
    protected $commonUtil;

    /**
     * Constructor
     *
     * @param Util $commonUtil
     * @return void
     */
    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    /**
     * Display a listing of the resource.
     * @return Response
     */
    public function index(Request $request)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;
            $is_closed = $request->input('account_status') == 'closed' ? 1 : 0;

            $accounts = Account::leftjoin('account_transactions as AT', function ($join) {
                $join->on('AT.account_id', '=', 'accounts.id');
                $join->whereNull('AT.deleted_at');
            })
                ->leftjoin('account_types as ats', 'accounts.account_type_id', '=', 'ats.id')
                ->leftjoin('account_types as pat', 'ats.parent_account_type_id', '=', 'pat.id')
                ->leftJoin('users AS u', 'accounts.created_by', '=', 'u.id')
                ->where('accounts.business_id', $business_id)
                ->where('is_closed', $is_closed)
                ->select([
                    'accounts.name',
                    'accounts.account_number',
                    'accounts.note',
                    'accounts.id',
                    'accounts.account_type_id',
                    'ats.name as account_type_name',
                    'pat.name as parent_account_type_name',
                    'is_closed',
                    DB::raw("SUM( IF(AT.type='credit', amount, -1*amount) ) as balance"),
                    DB::raw("CONCAT(COALESCE(u.surname, ''),' ',COALESCE(u.first_name, ''),' ',COALESCE(u.last_name,'')) as added_by")
                ])
                ->groupBy('accounts.id')
                ->get();

            $not_linked_payments = TransactionPayment::leftjoin(
                'transactions as T',
                'transaction_payments.transaction_id',
                '=',
                'T.id'
            )
                ->whereNull('transaction_payments.parent_id')
                ->where('transaction_payments.business_id', $business_id)
                ->whereNull('account_id')
                ->count();

            $account_types = AccountType::where('business_id', $business_id)
                ->whereNull('parent_account_type_id')
                ->with(['sub_types'])
                ->get();

            $data = [
                'accounts' => $accounts,
                'not_linked_payments' => $not_linked_payments,
                'account_types' => $account_types
            ];

            return $this->sendResponse($data, 'Accounts retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Show the form for creating a new resource (API version returns form data).
     * @return Response
     */
    public function create(Request $request)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;
            $account_types = AccountType::where('business_id', $business_id)
                ->whereNull('parent_account_type_id')
                ->with(['sub_types'])
                ->get();

            return $this->sendResponse(['account_types' => $account_types], 'Account creation form data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     * @param  Request $request
     * @return Response
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'account_number' => 'nullable|string|max:255',
            'note' => 'nullable|string',
            'account_type_id' => 'required|exists:account_types,id',
            'opening_balance' => 'nullable|numeric'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error.', $validator->errors(), 422);
        }

        try {
            $input = $request->only(['name', 'account_number', 'note', 'account_type_id']);
            $business_id = $request->user()->business_id;
            $user_id = $request->user()->id;
            $input['business_id'] = $business_id;
            $input['created_by'] = $user_id;

            $account = Account::create($input);

            //Opening Balance
            $opening_bal = $request->input('opening_balance');

            if (!empty($opening_bal)) {
                $ob_transaction_data = [
                    'amount' => $this->commonUtil->num_uf($opening_bal),
                    'account_id' => $account->id,
                    'type' => 'credit',
                    'sub_type' => 'opening_balance',
                    'operation_date' => Carbon::now(),
                    'created_by' => $user_id
                ];

                AccountTransaction::createAccountTransaction($ob_transaction_data);
            }

            return $this->sendResponse($account, 'Account created successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Show the specified resource.
     * @return Response
     */
    public function show(Request $request, $id)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $account = Account::where('business_id', $business_id)
                ->with(['account_type', 'account_type.parent_account'])
                ->findOrFail($id);

            // Get account transactions
            $transactions_query = AccountTransaction::join(
                'accounts as A',
                'account_transactions.account_id',
                '=',
                'A.id'
            )
                ->leftJoin('transaction_payments AS tp', 'account_transactions.transaction_payment_id', '=', 'tp.id')
                ->leftJoin('users AS u', 'account_transactions.created_by', '=', 'u.id')
                ->leftJoin('contacts AS c', 'tp.payment_for', '=', 'c.id')
                ->where('A.business_id', $business_id)
                ->where('A.id', $id)
                ->with(['transaction', 'transaction.contact', 'transfer_transaction', 'media', 'transfer_transaction.media'])
                ->select([
                    'account_transactions.type',
                    'account_transactions.amount',
                    'operation_date',
                    'sub_type',
                    'transfer_transaction_id',
                    DB::raw('(SELECT SUM(IF(AT.type="credit", AT.amount, -1 * AT.amount)) from account_transactions as AT WHERE AT.operation_date <= account_transactions.operation_date AND AT.account_id  =account_transactions.account_id AND AT.deleted_at IS NULL AND AT.id <= account_transactions.id) as balance'),
                    'account_transactions.transaction_id',
                    'account_transactions.id',
                    'tp.is_advance',
                    'tp.payment_ref_no',
                    'c.name as payment_for',
                    DB::raw("CONCAT(COALESCE(u.surname, ''),' ',COALESCE(u.first_name, ''),' ',COALESCE(u.last_name,'')) as added_by")
                ])
                ->groupBy('account_transactions.id')
                ->orderBy('account_transactions.id', 'asc')
                ->orderBy('account_transactions.operation_date', 'asc');

            if (!empty($request->input('type'))) {
                $transactions_query->where('account_transactions.type', $request->input('type'));
            }

            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');

            if (!empty($start_date) && !empty($end_date)) {
                $transactions_query->whereBetween(DB::raw('date(operation_date)'), [$start_date, $end_date]);
            }

            $transactions = $transactions_query->get();

            $data = [
                'account' => $account,
                'transactions' => $transactions
            ];

            return $this->sendResponse($data, 'Account details retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Account not found.', [], 404);
        }
    }

    /**
     * Show the form for editing the specified resource (API version returns edit form data).
     * @return Response
     */
    public function edit(Request $request, $id)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;
            $account = Account::where('business_id', $business_id)->find($id);

            if (!$account) {
                return $this->sendError('Account not found.', [], 404);
            }

            $account_types = AccountType::where('business_id', $business_id)
                ->whereNull('parent_account_type_id')
                ->with(['sub_types'])
                ->get();

            $data = [
                'account' => $account,
                'account_types' => $account_types
            ];

            return $this->sendResponse($data, 'Account edit form data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     * @param  Request $request
     * @return Response
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'account_number' => 'nullable|string|max:255',
            'note' => 'nullable|string',
            'account_type_id' => 'required|exists:account_types,id'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error.', $validator->errors(), 422);
        }

        try {
            $input = $request->only(['name', 'account_number', 'note', 'account_type_id']);
            $business_id = $request->user()->business_id;

            $account = Account::where('business_id', $business_id)->findOrFail($id);
            $account->update($input);

            return $this->sendResponse($account, 'Account updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Remove the specified account transaction from storage.
     * @return Response
     */
    public function destroyAccountTransaction(Request $request, $id)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $account_transaction = AccountTransaction::findOrFail($id);

            if (in_array($account_transaction->sub_type, ['fund_transfer', 'deposit'])) {
                //Delete transfer transaction for fund transfer
                if (!empty($account_transaction->transfer_transaction_id)) {
                    $transfer_transaction = AccountTransaction::findOrFail($account_transaction->transfer_transaction_id);
                    $transfer_transaction->delete();
                }
                $account_transaction->delete();
            }

            return $this->sendResponse([], 'Transaction deleted successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Closes the specified account.
     * @return Response
     */
    public function close(Request $request, $id)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $account = Account::where('business_id', $business_id)->findOrFail($id);
            $account->is_closed = 1;
            $account->save();

            return $this->sendResponse($account, 'Account closed successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Activates the specified account.
     * @return Response
     */
    public function activate(Request $request, $id)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $account = Account::where('business_id', $business_id)->findOrFail($id);
            $account->is_closed = 0;
            $account->save();

            return $this->sendResponse($account, 'Account activated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Shows form data for fund transfer.
     * @param  int $id
     * @return Response
     */
    public function getFundTransfer(Request $request, $id)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $from_account = Account::where('business_id', $business_id)
                ->NotClosed()
                ->find($id);

            if (!$from_account) {
                return $this->sendError('Account not found.', [], 404);
            }

            $to_accounts = Account::where('business_id', $business_id)
                ->where('id', '!=', $id)
                ->NotClosed()
                ->get(['id', 'name']);

            $data = [
                'from_account' => $from_account,
                'to_accounts' => $to_accounts
            ];

            return $this->sendResponse($data, 'Fund transfer data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Transfers fund from one account to another.
     * @return Response
     */
    public function postFundTransfer(Request $request)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0.01',
            'from_account' => 'required|exists:accounts,id',
            'to_account' => 'required|exists:accounts,id|different:from_account',
            'operation_date' => 'required|date',
            'note' => 'nullable|string'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error.', $validator->errors(), 422);
        }

        try {
            $business_id = $request->user()->business_id;

            $amount = $this->commonUtil->num_uf($request->input('amount'));
            $from = $request->input('from_account');
            $to = $request->input('to_account');
            $note = $request->input('note');

            if (!empty($amount)) {
                DB::beginTransaction();

                $debit_data = [
                    'amount' => $amount,
                    'account_id' => $from,
                    'type' => 'debit',
                    'sub_type' => 'fund_transfer',
                    'created_by' => $request->user()->id,
                    'note' => $note,
                    'transfer_account_id' => $to,
                    'operation_date' => $this->commonUtil->uf_date($request->input('operation_date'), true),
                ];

                $debit = AccountTransaction::createAccountTransaction($debit_data);

                $credit_data = [
                    'amount' => $amount,
                    'account_id' => $to,
                    'type' => 'credit',
                    'sub_type' => 'fund_transfer',
                    'created_by' => $request->user()->id,
                    'note' => $note,
                    'transfer_account_id' => $from,
                    'transfer_transaction_id' => $debit->id,
                    'operation_date' => $this->commonUtil->uf_date($request->input('operation_date'), true),
                ];

                $credit = AccountTransaction::createAccountTransaction($credit_data);

                $debit->transfer_transaction_id = $credit->id;
                $debit->save();

                Media::uploadMedia($business_id, $debit, $request, 'document');

                DB::commit();

                $data = [
                    'debit_transaction' => $debit,
                    'credit_transaction' => $credit
                ];

                return $this->sendResponse($data, 'Fund transferred successfully.');
            }
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Shows data for deposit.
     * @param  int $id
     * @return Response
     */
    public function getDeposit(Request $request, $id)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $account = Account::where('business_id', $business_id)
                ->NotClosed()
                ->find($id);

            if (!$account) {
                return $this->sendError('Account not found.', [], 404);
            }

            $from_accounts = Account::where('business_id', $business_id)
                ->where('id', '!=', $id)
                ->NotClosed()
                ->get(['id', 'name']);

            $data = [
                'account' => $account,
                'from_accounts' => $from_accounts
            ];

            return $this->sendResponse($data, 'Deposit data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Deposits amount.
     * @param  Request $request
     * @return json
     */
    public function postDeposit(Request $request)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0.01',
            'account_id' => 'required|exists:accounts,id',
            'operation_date' => 'required|date',
            'note' => 'nullable|string',
            'from_account' => 'nullable|exists:accounts,id|different:account_id'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error.', $validator->errors(), 422);
        }

        try {
            $business_id = $request->user()->business_id;

            $amount = $this->commonUtil->num_uf($request->input('amount'));
            $account_id = $request->input('account_id');
            $note = $request->input('note');

            $account = Account::where('business_id', $business_id)->findOrFail($account_id);

            if (!empty($amount)) {
                $credit_data = [
                    'amount' => $amount,
                    'account_id' => $account_id,
                    'type' => 'credit',
                    'sub_type' => 'deposit',
                    'operation_date' => $this->commonUtil->uf_date($request->input('operation_date'), true),
                    'created_by' => $request->user()->id,
                    'note' => $note
                ];
                $credit = AccountTransaction::createAccountTransaction($credit_data);

                $from_account = $request->input('from_account');
                if (!empty($from_account)) {
                    $debit_data = $credit_data;
                    $debit_data['type'] = 'debit';
                    $debit_data['account_id'] = $from_account;
                    $debit_data['transfer_transaction_id'] = $credit->id;

                    $debit = AccountTransaction::createAccountTransaction($debit_data);

                    $credit->transfer_transaction_id = $debit->id;
                    $credit->save();
                }

                $data = [
                    'credit_transaction' => $credit,
                    'debit_transaction' => $debit ?? null
                ];

                return $this->sendResponse($data, 'Amount deposited successfully.');
            }
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Calculates account current balance.
     * @param  int $id
     * @return json
     */
    public function getAccountBalance(Request $request, $id)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;
            $account = Account::leftjoin(
                'account_transactions as AT',
                'AT.account_id',
                '=',
                'accounts.id'
            )
                ->whereNull('AT.deleted_at')
                ->where('accounts.business_id', $business_id)
                ->where('accounts.id', $id)
                ->select('accounts.*', DB::raw("SUM( IF(AT.type='credit', amount, -1 * amount) ) as balance"))
                ->first();

            if (!$account) {
                return $this->sendError('Account not found.', [], 404);
            }

            return $this->sendResponse($account, 'Account balance retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Show the cash flow.
     * @return Response
     */
    public function cashFlow(Request $request)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $accounts_query = AccountTransaction::join(
                'accounts as A',
                'account_transactions.account_id',
                '=',
                'A.id'
            )
                ->leftjoin(
                    'transaction_payments as TP',
                    'account_transactions.transaction_payment_id',
                    '=',
                    'TP.id'
                )
                ->where('A.business_id', $business_id)
                ->with(['transaction', 'transaction.contact', 'transfer_transaction'])
                ->select([
                    'type',
                    'account_transactions.amount',
                    'operation_date',
                    'sub_type',
                    'transfer_transaction_id',
                    DB::raw("(SELECT SUM(IF(AT.type='credit', AT.amount, -1 * AT.amount)) from account_transactions as AT JOIN accounts as ac ON ac.id=AT.account_id WHERE ac.business_id= $business_id AND AT.operation_date <= account_transactions.operation_date AND AT.deleted_at IS NULL) as balance"),
                    'account_transactions.transaction_id',
                    'account_transactions.id',
                    'A.name as account_name',
                    'TP.payment_ref_no as payment_ref_no'
                ])
                ->groupBy('account_transactions.id')
                ->orderBy('account_transactions.operation_date', 'desc');

            if (!empty($request->input('type'))) {
                $accounts_query->where('type', $request->input('type'));
            }

            if (!empty($request->input('account_id'))) {
                $accounts_query->where('A.id', $request->input('account_id'));
            }

            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');

            if (!empty($start_date) && !empty($end_date)) {
                $accounts_query->whereBetween(DB::raw('date(operation_date)'), [$start_date, $end_date]);
            }

            $transactions = $accounts_query->get();
            $accounts = Account::forDropdown($business_id, false);

            $data = [
                'transactions' => $transactions,
                'accounts' => $accounts
            ];

            return $this->sendResponse($data, 'Cash flow retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Get payment details for transaction
     * @param  object $row
     * @return string
     */
    private function __getPaymentDetails($row)
    {
        $details = '';
        if (!empty($row->sub_type)) {
            $details = __('account.' . $row->sub_type);
            if (in_array($row->sub_type, ['fund_transfer', 'deposit']) && !empty($row->transfer_transaction)) {
                if ($row->type == 'credit') {
                    $details .= ' ( ' . __('account.from') . ': ' . $row->transfer_transaction->account->name . ')';
                } else {
                    $details .= ' ( ' . __('account.to') . ': ' . $row->transfer_transaction->account->name . ')';
                }
            }
        } else {
            if (!empty($row->transaction->type)) {
                if ($row->transaction->type == 'purchase') {
                    $details = __('lang_v1.purchase') . ' - ' . __('purchase.supplier') . ': ' . $row->transaction->contact->name . ' - ' .
                        __('purchase.ref_no') . ': ' . $row->transaction->ref_no;
                } elseif ($row->transaction->type == 'expense') {
                    $details = __('lang_v1.expense') . ' - ' . __('purchase.ref_no') . ': ' . $row->transaction->ref_no;
                } elseif ($row->transaction->type == 'sell') {
                    $details = __('sale.sale') . ' - ' . __('contact.customer') . ': ' . $row->transaction->contact->name . ' - ' .
                        __('sale.invoice_no') . ': ' . $row->transaction->invoice_no;
                }
            }
        }

        if (!empty($row->payment_ref_no)) {
            if (!empty($details)) {
                $details .= ' - ';
            }
            $details .= __('lang_v1.pay_reference_no') . ': ' . $row->payment_ref_no;
        }
        if (!empty($row->payment_for)) {
            if (!empty($details)) {
                $details .= ' - ';
            }
            $details .= __('account.payment_for') . ': ' . $row->payment_for;
        }

        if ($row->is_advance == 1) {
            $details .= ' (' . __('lang_v1.advance_payment') . ')';
        }

        return $details;
    }
}
