<?php

namespace App\Exports;

use App\Models\BusinessLocation;
use App\Models\Product;
use App\Models\VariationLocationDetail;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithEvents;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use Maatwebsite\Excel\Events\AfterSheet;

class InventoryExport implements FromCollection, WithHeadings, WithMapping, WithStyles, WithColumnWidths, WithEvents
{
    protected $location_id;
    protected $row_count = 0;

    public function __construct($location_id)
    {
        $this->location_id = $location_id;
    }

    public function collection()
    {
        $data = VariationLocationDetail::with(['product', 'location'])
            ->leftJoin('products as p', 'p.id', '=', 'variation_location_details.product_id')
            ->leftJoin('business_locations as bl', 'bl.id', '=', 'variation_location_details.location_id')
            ->leftJoin('variations as v', 'v.product_id', '=', 'p.id')
            ->where('bl.id', $this->location_id)
            ->select(
                'p.id as product_id',
                'p.name as name',
                'bl.id as location_id',
                'bl.name as location',
                'variation_location_details.qty_available as qty_available',
                'v.default_sell_price as sell_price'
            )
            ->get();

        $this->row_count = $data->count() + 1;
        return $data;
    }

    public function headings(): array
    {
        return [
            'EMPLACEMENT',
            'PRODUIT',
            'STOCK THÉORIQUE',
            'STOCK PHYSIQUE',
            'ÉCART',
            'PRIX',
            'TOTAL STOCK THÉORIQUE',
            'TOTAL STOCK PHYSIQUE',
            'ÉCART TOTAL',
            'OBSERVATION',
            'PRODUCT ID',
            'LOCATION ID'
        ];
    }

    public function map($row): array
    {
        static $rowNumber = 2;

        $mapped = [
            $row->location,
            $row->name,
            $row->qty_available,
            '', // Stock physique (à remplir)
            '=D' . $rowNumber . '-C' . $rowNumber, // Écart
            $row->sell_price,
            '=C' . $rowNumber . '*F' . $rowNumber, // Total théorique
            '=D' . $rowNumber . '*F' . $rowNumber, // Total physique
            '=H' . $rowNumber . '-G' . $rowNumber, // Écart total
            '', // Observation
            $row->product_id,
            $row->location_id
        ];

        $rowNumber++;
        return $mapped;
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => [
                'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4F81BD']
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER
                ]
            ],
            'A1:L' . $this->row_count => [
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => ['rgb' => '000000']
                    ]
                ],
                'alignment' => [
                    'vertical' => Alignment::VERTICAL_CENTER
                ]
            ],
            // Ajout d'un style spécifique pour la colonne du stock théorique
            'C2:C' . $this->row_count => [
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'E2EFDA']  // Fond vert clair pour indiquer que c'est protégé
                ]
            ]
        ];
    }

    public function columnWidths(): array
    {
        return [
            'A' => 20, // EMPLACEMENT
            'B' => 40, // PRODUIT
            'C' => 15, // STOCK THÉORIQUE
            'D' => 15, // STOCK PHYSIQUE
            'E' => 15, // ÉCART
            'F' => 12, // PRIX
            'G' => 20, // TOTAL STOCK THÉORIQUE
            'H' => 20, // TOTAL STOCK PHYSIQUE
            'I' => 15, // ÉCART TOTAL
            'J' => 30, // OBSERVATION
            'K' => 10, // PRODUCT ID
            'L' => 10  // LOCATION ID
        ];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                // Activer la protection de la feuille
                $protection = $event->sheet->getProtection();
                $protection->setSheet(true);
                $protection->setPassword(Auth::user()->username);
                $protection->setSelectLockedCells(false);
                $protection->setSelectUnlockedCells(true);

                // Masquer les colonnes ID
                $event->sheet->getColumnDimension('K')->setVisible(false); // Product ID
                $event->sheet->getColumnDimension('L')->setVisible(false); // Location ID

                // Verrouiller toutes les cellules par défaut
                $event->sheet->getStyle('A1:L' . $this->row_count)
                    ->getProtection()
                    ->setLocked(\PhpOffice\PhpSpreadsheet\Style\Protection::PROTECTION_PROTECTED);

                // Protection spécifique pour le stock théorique
                $event->sheet->getStyle('C2:C' . $this->row_count)
                    ->getProtection()
                    ->setLocked(\PhpOffice\PhpSpreadsheet\Style\Protection::PROTECTION_PROTECTED);

                // Déverrouiller uniquement les cellules pour la saisie
                $event->sheet->getStyle('D2:D' . $this->row_count) // Stock physique
                    ->getProtection()
                    ->setLocked(\PhpOffice\PhpSpreadsheet\Style\Protection::PROTECTION_UNPROTECTED);

                $event->sheet->getStyle('J2:J' . $this->row_count) // Observations
                    ->getProtection()
                    ->setLocked(\PhpOffice\PhpSpreadsheet\Style\Protection::PROTECTION_UNPROTECTED);

                // Validation des données pour le stock physique
                for ($i = 2; $i <= $this->row_count; $i++) {
                    $validation = $event->sheet->getCell('D' . $i)->getDataValidation();
                    $validation->setType(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::TYPE_WHOLE);
                    $validation->setErrorStyle(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::STYLE_STOP);
                    $validation->setAllowBlank(false);
                    $validation->setShowInputMessage(true);
                    $validation->setShowErrorMessage(true);
                    $validation->setErrorTitle('Erreur de saisie');
                    $validation->setError('Veuillez entrer un nombre entier positif');
                    $validation->setPromptTitle('Stock physique');
                    $validation->setPrompt('Entrez le stock physique compté');
                    $validation->setFormula1(0);
                    $validation->setOperator(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::OPERATOR_GREATERTHANOREQUAL);
                }
            }
        ];
    }
}
